package ftbsc.lll.processor.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * Marks a method as the target method.
 * The method annotated with this, called "stub" within the documentation, should have the
 * same name and parameters as the method it's supposed to represent.
 * It will be discarded unless the containing class is annotated with {@link Patch}
 * and at least another method within the class is annotated with {@link Injector}.
 * @see Patch
 * @see Injector
 */
@Retention(RetentionPolicy.CLASS)
@Repeatable(Target.List.class)
@java.lang.annotation.Target(ElementType.METHOD)
public @interface Target {

	/**
	 * Indicates which of the methods annotated with {@link Find} or {@link Injector}
	 * is targeting this stub.
	 * @return the name of the element this is supposed to apply to
	 * @since 0.5.0
	 */
	String of();

	/**
	 * Provides a name to be used instead of the stub's.
	 * @return a name which overrides the name of the annotated one, may be used in
	 * 				 cases such as constructors or ambiguous functions
	 * @since 0.5.0
	 */
	String methodName() default "";

	/**
	 * When set to true, tells the processor to avoid any attempts at verifying the existence of the
	 * given method. This will still attempt to process mappings for the method and its arguments.
	 * Do note that this should only be used as a last resort when no other option is available,
	 * as this can potentially lead to runtime errors. This is useful, for instance, for patching
	 * lambdas in newer versions of Java.
	 * If this is ture, the finder will honor {@link #methodName()}, but ignore all other modifiers
	 * (like {@link #strict()}).
	 * @return whether unchecked mode is to be used
	 * @since 0.9.0
	 */
	boolean unchecked() default false;

	/**
	 * When set to false, tells the processor to first try to match a single method by name,
	 * and to only check parameters if further clarification is needed.
	 * While non-strict mode is more computationally efficient, it's ultimately not
	 * relevant, as it only matters at compile time. Do not set this to false unless
	 * you know what you're doing.
	 * @return whether strict mode is to be used
	 * @since 0.3.0
	 */
	boolean strict() default true;

	/**
	 * When set to true, tells the processor to try to match the synthetic "bridge" method
	 * generated by the compiler to handle type erasure.
	 * @return whether the bridge method should be targeted instead of the actual method
	 * @since 0.5.2
	 */
	boolean bridge() default false;

	/**
	 * Used to support {@link Target} as a {@link Repeatable} annotation.
	 * @since 0.6.1
	 */
	@Retention(RetentionPolicy.CLASS)
	@java.lang.annotation.Target(ElementType.METHOD)
	@interface List {
		/**
		 * @return the {@link Injector} annotations, as an array
		 */
		Target[] value();
	}
}
